/**
 * Copyright 2010 FH Trier, Umwelt-Campus Birkenfeld
 *
 * FH Trier, Umwelt-Campus Birkenfeld
 * P.O. Box 1380
 * D-55761 Birkenfeld
 * GERMANY
 *
 * http://www.umwelt-campus.de
 *
 * Lizenziert unter der EUPL, Version 1.1 oder - sobald
 * diese von der Europäischen Kommission genehmigt wurden -
 * Folgeversionen der EUPL ("Lizenz");
 * Sie dürfen dieses Werk ausschließlich gemäß dieser Lizenz nutzen.
 * Eine Kopie der Lizenz finden Sie hier:
 *
 * http://ec.europa.eu/idabc/eupl
 *
 * Sofern nicht durch anwendbare Rechtsvorschriften
 * gefordert oder in schriftlicher Form vereinbart, wird
 * die unter der Lizenz verbreitete Software "so wie sie ist",
 * OHNE JEGLICHE GEWÄHRLEISTUNG ODER BEDINGUNGEN -
 * ausdrücklich oder stillschweigend - verbreitet.
 * Die sprachspezifischen Genehmigungen und Beschränkungen
 * unter der Lizenz sind dem Lizenztext zu entnehmen.
 */
package de.umweltcampus.uput.informatik.openeanvadapter.communication.adapterservices;

import de.umweltcampus.uput.informatik.openeanvadapter.communication.osci.OSCIProxy;
import de.umweltcampus.uput.informatik.openeanvadapter.data.dao.AddressDAO;
import de.umweltcampus.uput.informatik.openeanvadapter.data.dao.CompanyDAO;
import de.umweltcampus.uput.informatik.openeanvadapter.data.dao.MessageDAO;
import de.umweltcampus.uput.informatik.openeanvadapter.data.model.Address;
import de.umweltcampus.uput.informatik.openeanvadapter.data.model.Certificate;
import de.umweltcampus.uput.informatik.openeanvadapter.data.model.Company;
import de.umweltcampus.uput.informatik.openeanvadapter.data.model.Message;
import de.umweltcampus.uput.informatik.openeanvadapter.util.common.CommonException;
import de.umweltcampus.uput.informatik.openeanvadapter.util.common.Profil;
import java.io.UnsupportedEncodingException;
import java.lang.String;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.jws.WebMethod;
import javax.jws.WebParam;
import javax.jws.WebService;
import javax.jws.soap.SOAPBinding;
import javax.jws.soap.SOAPBinding.Style;

/**
 * Stellt die Web Service-Schnittstellen für andere Softwaresysteme bereit
 * @author Timo Johann & Marc Beck
 */
@WebService(name = "AdapterWebService")
@SOAPBinding(style = Style.RPC)
public class AdapterWebService {

    private OSCIProxy messageService;

    public AdapterWebService() {
        this.messageService = new OSCIProxy();
    }

    /**
     * Sendet eine Nachricht mit einer gegebenen Profilierung über den im
     * Adapter integrierten Webservice.
     * <br>
     * Die in der Profilierung angegebenen Adressen müssen inkl. Zertifikate
     * im Adapter vorhanden sein.
     * @
     * @param senderID
     * @param receiverID
     * @param profilierung
     * @param nachricht
     * @return Erfolgs-/Fehlernachricht
     */
    @WebMethod
    public String sendMessage(@WebParam(name = "senderID") Long senderID, @WebParam(name = "receiverID") Long receiverID, @WebParam(name = "profil") String profilierung, @WebParam(name = "nachricht") String nachricht) {
        String response = "";
        try {
            Profil profil = new Profil(profilierung);
            AddressDAO dao = new AddressDAO();
            Address sender = dao.findById(senderID);
            Address receiver = dao.findById(receiverID);
            Certificate certificate = receiver.getPublicCertificate();
            response = messageService.send(sender, certificate.getCertificate(), profil, nachricht.getBytes("UTF-8"));
        } catch (UnsupportedEncodingException ex) {
            Logger.getLogger(AdapterWebService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (CommonException ex) {
            Logger.getLogger(AdapterWebService.class.getName()).log(Level.SEVERE, null, ex);
        }
        return response;
    }

    /**
     * Fragt die Namen aller Unternehmen an, die im Adapter hinterlegt sind
     * @return
     */
    public String[] getCompanyNames() {
        CompanyDAO cdao = new CompanyDAO();
        String[] companyNames = null;
        List<Company> companies = cdao.findByNamedQuery("Company.findAll");
        if (!companies.isEmpty()) {
            companyNames = new String[companies.size()];
            for (int i = 0; i < companyNames.length; i++) {
                companyNames[i] = companies.get(i).getName();


            }
           
        }
        return companyNames;
    }

    /**
     * Gibt die IDs aller Unternehmensadressen die zum Unternehmen mit dem Namen zurück
     * @param Der Name des Unternehmens wie er im Adapter hinterlefgt ist
     * @return ID der Adressen + Name1
     */
    @WebMethod
    public AwsCompanyAddress[] getCompanyAddresses(@WebParam(name = "companyName") String companyName) {
        AwsCompanyAddress[] addresses = null;
        CompanyDAO cdao = new CompanyDAO();
        List<Company> companies = cdao.findByNamedQuery("Company.findByName", companyName);
        if (!companies.isEmpty()) {
            Company c = companies.get(0);
            List<Address> list = c.getAddresses();
            addresses = new AwsCompanyAddress[list.size()];
            for (int i = 0; i < addresses.length; i++) {
                Address a = list.get(i);
                addresses[i] = new AwsCompanyAddress(a.getId(), a.getName1());
            }
        }
        return addresses;

    }

    /**
     * Eine Liste aller nicht abgeholten Nachrichten
     * @return Subjektstrings der Nachrichten
     */
    @WebMethod
    public String[] getMessageList(@WebParam(name = "receiverID") Long receiverID) {
        ArrayList<String> messageList = new ArrayList<String>();
        AddressDAO adao = new AddressDAO();
        Address address = adao.findById(receiverID);
        List<Message> messages = address.getReceivedMessages();
        for (Message message : messages) {
            if (!message.isRetrieved()) {
                messageList.add(message.getSubjectString());
            }
        }
        String[] list = new String[messageList.size()];
        for (int i = 0; i < list.length; i++) {
            list[i] = messageList.get(i);
        }
        return list;
    }

    /**
     * Gibt die zu den Profilen zugehörigen Nachrichten zurück
     * @param profiles  Eine Liste der Nachrichtenprofile der Nachrichten die abgeholt werden sollen
     * @return Nachrichten
     */
    public String[] receiveMessages(@WebParam(name = "profiles") String[] profiles) {
        ArrayList<String> messages = new ArrayList<String>();
        MessageDAO mdao = new MessageDAO();
        for (String profile : profiles) {
            Message message = (Message) mdao.findByNamedQuery("Message.findByProfile", profile);
            message.setRetrieved(true);
            mdao.edit(message);
            messages.add(message.getMessage().toString());
        }
        String[] list = new String[messages.size()];
        for (int i = 0; i < list.length; i++) {
            list[i] = messages.get(i);
        }
        return list;
    }

    /**
     * Empfängt alle Nachrichten, die noch nicht abgeholt worden sind.
     * @param receiverID
     * @return
     */
    @WebMethod
    public String[] receiveAllMessages(@WebParam(name = "receiverID") Long receiverID) {
        ArrayList<String> messages = new ArrayList<String>();
        AddressDAO dao = new AddressDAO();
        MessageDAO messageDAO = new MessageDAO();
        Address receiver = dao.findById(receiverID);
        List<Message> messageList = receiver.getReceivedMessages();
        for (Message message : messageList) {
            if (!message.isRetrieved()) {
                messages.add(new String(message.getMessage()));
                message.setRetrieved(true);
                messageDAO.edit(message);
            }
        }
        String[] list = new String[messages.size()];
        for (int i = 0; i < list.length; i++) {
            list[i] = messages.get(i);
        }
        return list;
    }
}
