/**
 * Copyright 2010 FH Trier, Umwelt-Campus Birkenfeld
 * 
 * FH Trier, Umwelt-Campus Birkenfeld  
 * P.O. Box 1380
 * D-55761 Birkenfeld
 * GERMANY
 *
 * http://www.umwelt-campus.de
 *
 * Lizenziert unter der EUPL, Version 1.1 oder - sobald
 * diese von der Europäischen Kommission genehmigt wurden -
 * Folgeversionen der EUPL ("Lizenz");
 * Sie dürfen dieses Werk ausschließlich gemäß dieser Lizenz nutzen.
 * Eine Kopie der Lizenz finden Sie hier:
 *
 * http://ec.europa.eu/idabc/eupl
 *
 * Sofern nicht durch anwendbare Rechtsvorschriften
 * gefordert oder in schriftlicher Form vereinbart, wird
 * die unter der Lizenz verbreitete Software "so wie sie ist",
 * OHNE JEGLICHE GEWÄHRLEISTUNG ODER BEDINGUNGEN -
 * ausdrücklich oder stillschweigend - verbreitet.
 * Die sprachspezifischen Genehmigungen und Beschränkungen
 * unter der Lizenz sind dem Lizenztext zu entnehmen.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package de.umweltcampus.uput.informatik.openeanvadapter.communication.osci;

import de.osci.osci12.OSCIException;
import de.osci.osci12.common.DialogHandler;
import de.osci.osci12.messageparts.Content;
import de.osci.osci12.messageparts.ContentContainer;
import de.osci.osci12.messageparts.ProcessCardBundle;
import de.osci.osci12.messagetypes.ExitDialog;
import de.osci.osci12.messagetypes.FetchDelivery;
import de.osci.osci12.messagetypes.FetchProcessCard;
import de.osci.osci12.messagetypes.GetMessageId;
import de.osci.osci12.messagetypes.InitDialog;
import de.osci.osci12.messagetypes.OSCIMessage;
import de.osci.osci12.messagetypes.ResponseToExitDialog;
import de.osci.osci12.messagetypes.ResponseToFetchDelivery;
import de.osci.osci12.messagetypes.ResponseToFetchProcessCard;
import de.osci.osci12.messagetypes.ResponseToGetMessageId;
import de.osci.osci12.messagetypes.ResponseToInitDialog;
import de.osci.osci12.messagetypes.ResponseToStoreDelivery;
import de.osci.osci12.messagetypes.StoreDelivery;
import de.osci.osci12.roles.Addressee;
import de.osci.osci12.roles.Intermed;
import de.osci.osci12.roles.Originator;
import de.osci.osci12.samples.impl.HttpTransport;
import de.osci.osci12.samples.impl.crypto.PKCS12Decrypter;
import de.osci.osci12.samples.impl.crypto.PKCS12Signer;
import de.umweltcampus.uput.informatik.openeanvadapter.communication.zksservices.ZKSMessageUtil;
import de.umweltcampus.uput.informatik.openeanvadapter.communication.logging.ZKSMessageLogContainer;
import de.umweltcampus.uput.informatik.openeanvadapter.communication.logging.ZKSMessageLogger;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import de.umweltcampus.uput.informatik.openeanvadapter.resources.MessageBundle;

/**
 * OSCIMessageService
 * @author Marc Beck
 */
public class OSCIMessageService {

    private final OSCIErrorLogger errorLogger = new OSCIErrorLogger();
    private Intermed intermed;
    private ZKSMessageLogger zKSMessageLogger;

    /**
     * Konstruktor
     * @param intermedCert
     * @param intermedAdress
     */
    OSCIMessageService(byte[] intermedCert, String intermedAdress) {
        zKSMessageLogger = new ZKSMessageLogger();
        try {
            java.security.cert.X509Certificate intermedCipherCert = de.osci.helper.Tools.createCertificate(intermedCert);
            intermed = new Intermed(null, intermedCipherCert, new URI(intermedAdress));
        } catch (CertificateException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (URISyntaxException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    /**
     * Sendet eine Nachricht vom Sender an den Empfäger
     * @param logData OSCILogData für den Sender
     * @param receiverCert das öffentliche Zertifikat des Empfängers
     * @param subject der Betreff der Nachricht (Profilierung)
     * @param message die Nachricht selbst
     * @return
     */
    String send(OSCILogData logData, byte[] receiverCert, String subject, byte[] message) {
        InputStream is = null;
        InputStream is1 = null;
        try {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.INFO, "Nachricht senden gestartet");
            // Erzeugen des Originators.
            // Dazu müssen die Privaten Zertifikate verwendet werden.
            is = new ByteArrayInputStream(logData.getAdresse().getPrivateCertificate().getCertificate());
            is1 = new ByteArrayInputStream(logData.getAdresse().getPrivateCertificate().getCertificate());
            Originator origi = new Originator(new PKCS12Signer(is, logData.getAdresse().getPrivateCertificate().getPassword()), new PKCS12Decrypter(is1, logData.getAdresse().getPrivateCertificate().getPassword()));
            // DialogHandler erzeugen
            DialogHandler clientDialog = new DialogHandler(origi, intermed, new de.osci.osci12.samples.impl.HttpTransport());
            // Die GetMessageId-Anfrage erzeugen
            GetMessageId getMsgID = new GetMessageId(clientDialog);
            ResponseToGetMessageId rsp2GetMsgID = null;
            rsp2GetMsgID = getMsgID.send();
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.INFO, "MsgID: " + rsp2GetMsgID.getMessageId());
            logData.appendBasicOSCIMessage("rsp2GetMsgID", rsp2GetMsgID.toString());
            //Empfänger Zertifikat suchen
            Addressee adressee = new Addressee(null, de.osci.helper.Tools.createCertificate(receiverCert));
            // Create the StoreDelivery-message object
            StoreDelivery storeDel = new StoreDelivery(clientDialog, adressee, rsp2GetMsgID.getMessageId());
            // Subject (Profil) setzen
            storeDel.setSubject(subject);
            // Dem Timestamp des Intermediärs vertrauen
            storeDel.setQualityOfTimeStampCreation(false);
            storeDel.setQualityOfTimeStampReception(false);
            // ContentContainer hinzufügen
            ContentContainer contentContainer = new ContentContainer();
            // Einen nicht-codierten UTF-8 String verwenden
            String messageUTF8 = new String(message, "UTF-8");
            contentContainer.addContent(new Content(messageUTF8));
            //(auf Nachrichtenebene keine Verschlüsselung)
            storeDel.addContentContainer(contentContainer);
            logData.appendBasicOSCIMessage("storeDel", storeDel.toString());
            // Anfrage abschicken und Antwort erhalten
            ResponseToStoreDelivery rsp2StoreDel = storeDel.send();
            logData.appendBasicOSCIMessage("rsp2StoreDel", rsp2StoreDel.toString());
            String[][] seses = rsp2StoreDel.getFeedback();
            if (seses[0][1].startsWith("0")) {
                Logger.getLogger(OSCIMessageService.class.getName()).log(Level.INFO, "Nachricht gesendet...");
            }
            //Nachricht loggen
            String messageType = ZKSMessageUtil.detectMessageType(messageUTF8);
            zKSMessageLogger.logGesendeteNachricht(message, messageType, logData.getAdresse(), subject, logData.getOsciMessages());
            //Fertig
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.INFO, "Nachricht senden beendet");
            return MessageBundle.getString("msgMessageTransmitted");
        } catch (OSCIException ex) {
            this.errorLogger.logOSCIError(new Date(), ex.getErrorCode(), ex.getLocalizedMessage(), subject);
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
            return ex.toString();
        } catch (KeyStoreException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
            return ex.toString();
        } catch (CertificateException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
            return ex.toString();
        } catch (NoSuchAlgorithmException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
            return ex.toString();
        } catch (IOException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
            return ex.toString();
        } catch (UnrecoverableKeyException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
            return ex.toString();
        } finally {
            try {
                is.close();
                is1.close();
            } catch (IOException ex) {
                //nix machen
            }
        }
    }

    /**
     * Holt alle Nachrichten für den spezifizierten Empfänger ab.
     * @param logData 
     * @return
     */
    String receive(OSCILogData logData) {
        try {

            List<ZKSMessageLogContainer> logContainers = new ArrayList<ZKSMessageLogContainer>();

            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.INFO, "Nachrichten abrufen gestartet");

            // Erzeugen des Originators.
            // Dazu müssen die Privaten Zertifikate verwendet werden.
            InputStream is = new ByteArrayInputStream(logData.getAdresse().getPrivateCertificate().getCertificate());
            InputStream is1 = new ByteArrayInputStream(logData.getAdresse().getPrivateCertificate().getCertificate());
            Originator origi = new Originator(new PKCS12Signer(is, logData.getAdresse().getPrivateCertificate().getPassword()), new PKCS12Decrypter(is1, logData.getAdresse().getPrivateCertificate().getPassword()));
            // DialogHandler erzeugen
            DialogHandler clientDialog = new DialogHandler(origi, intermed, new HttpTransport());
            // InitDialog-Anfrage erzeugen
            InitDialog id = new InitDialog(clientDialog);
            // Anfrage abschicken und Antwort erhalten
            ResponseToInitDialog rsp2InitDialog = null;
            rsp2InitDialog = id.send();

            // Der Dialog ist initialisiert.
            // Nun können wir den FetchProcessCard-request abschicken
            // FetchProcessCard
            FetchProcessCard fetchProcCard = new FetchProcessCard(clientDialog);
            // Rolle für Abrufen der Laufzettel bei uns nach Adressat
            fetchProcCard.setRoleForSelection(OSCIMessage.SELECT_ADDRESSEE);
            // Selektieren nach letztem Abruf-Zeitpunkt
            fetchProcCard.setSelectionMode(OSCIMessage.SELECT_BY_DATE_OF_RECEPTION);
            //Timestamp für die Adresse (wird später noch verwendet)
            String timestamp = logData.getAdresse().getOsciTimestamp();
            fetchProcCard.setSelectionRule(timestamp);
            // Anfrage abschicken und Antwort erhalten
            ResponseToFetchProcessCard rsp2FetchProcCard = fetchProcCard.send();
            ProcessCardBundle[] pBundle = rsp2FetchProcCard.getProcessCardBundles();
            int anzahlNachrichten = pBundle.length;
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.INFO, "Anzahl der abzuholenden Nachrichten: " + anzahlNachrichten);
            if (anzahlNachrichten == 0) {
                return "Keine Nachrichten vorhanden!";
            }
            for (ProcessCardBundle processCardBundle : pBundle) {
                if (processCardBundle.getReception() == null) {
                    ZKSMessageLogContainer logContainer = new ZKSMessageLogContainer();
                    FetchDelivery fetchDel = new FetchDelivery(clientDialog);
                    fetchDel.setSelectionMode(OSCIMessage.SELECT_BY_MESSAGE_ID);
                    fetchDel.setSelectionRule(processCardBundle.getMessageId());
                    ResponseToFetchDelivery rsp2FetchDel = fetchDel.send();
                    logContainer.setoSCIMessage(rsp2FetchDel.toString());
                    if (rsp2FetchDel.getContentContainer().length > 0) {
                        String content = rsp2FetchDel.getContentContainer()[0].getContents()[0].getContentData();
                        logContainer.setMessage(content.getBytes("UTF-8"));
                        //Nachrichtentyp ermitteln und empfangene Nachricht loggen
                        String msgType = ZKSMessageUtil.detectMessageType(content);
                        logContainer.setMessagetype(msgType);
                        logContainer.setSubject(rsp2FetchDel.getSubject());
                        //Timestamp setzen
                        timestamp = rsp2FetchDel.getTimestampForwarding().getTimeStamp();
                        logData.getAdresse().setOsciTimestamp(timestamp);
                        logContainers.add(logContainer);
                    }
                }
            }
            // Cleanup
            ExitDialog ed = new ExitDialog(clientDialog);
            ResponseToExitDialog rsp2ExitDialog = ed.send();

            //Logging für alle abgerufenen Nachrichten
            for (ZKSMessageLogContainer container : logContainers) {
                OSCILogData data = new OSCILogData(logData.getAdresse());
                data.appendBasicOSCIMessage("rsp2InitDialog", rsp2InitDialog.toString());
                data.appendBasicOSCIMessage("rsp2FetchDel", container.getoSCIMessage());
                data.appendBasicOSCIMessage("rsp2ExitDialog", rsp2ExitDialog.toString());
                zKSMessageLogger.logEmpfangeneNachricht(container.getMessage(), container.getMessageType(), data.getAdresse(), container.getSubject(), data.getOsciMessages());
            }

            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.INFO, "Nachrichten abrufen beendet");
            if (logContainers.size() == 0) {
                return "Keine Nachrichten vorhanden!";
            }
            return logContainers.size() + " Nachricht(en) erfolgreich abgerufen!";
        } catch (OSCIException ex) {
            this.errorLogger.logOSCIError(new Date(), ex.getErrorCode(), ex.getLocalizedMessage(), "");
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
            return ex.toString();
        } catch (KeyStoreException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
            return ex.toString();
        } catch (CertificateException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
            return ex.toString();
        } catch (NoSuchAlgorithmException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
            return ex.toString();
        } catch (IOException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
            return ex.toString();
        } catch (Exception ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
            return ex.toString();
        }
    }

    /**
     * Empfängt eine einzige Nachricht anhand ihrer ID.
     * @param logData 
     * @param messageId
     * @return
     */
    public byte[] receiveMessageById(OSCILogData logData, String messageId) {
        try {
            InputStream stream1 = new ByteArrayInputStream(logData.getAdresse().getPrivateCertificate().getCertificate());
            InputStream stream2 = new ByteArrayInputStream(logData.getAdresse().getPrivateCertificate().getCertificate());
            Originator origi = new Originator(new PKCS12Signer(stream1, logData.getAdresse().getPrivateCertificate().getPassword()), new PKCS12Decrypter(stream2, logData.getAdresse().getPrivateCertificate().getPassword()));
            DialogHandler clientDialog = new DialogHandler(origi, intermed, new HttpTransport());
            InitDialog id = new InitDialog(clientDialog);
            ResponseToInitDialog rsp2InitDialog = null;
            rsp2InitDialog = id.send();
            if (!rsp2InitDialog.getFeedback()[0][1].startsWith("0")) {
                // Error handling
                return null;
            }
            FetchDelivery fetchDel = new FetchDelivery(clientDialog);
            fetchDel.setSelectionMode(OSCIMessage.SELECT_BY_MESSAGE_ID);
            fetchDel.setSelectionRule(messageId);
            // Anfrage abschicken und Antwort erhalten
            ResponseToFetchDelivery rsp2FetchDel = fetchDel.send();
            if (!rsp2FetchDel.getFeedback()[0][1].startsWith("0")) {
                // Error handling
                return null;
            }
            Content content = rsp2FetchDel.getContentContainer()[0].getContents()[0];
            byte[] contentOfMessage = content.getContentData().getBytes("UTF-8");
            //Exit
            ExitDialog ed = new ExitDialog(clientDialog);
            ResponseToExitDialog rsp2ExitDialog = ed.send();

            //Logging
            String msgType = ZKSMessageUtil.detectMessageType(new String(contentOfMessage, "UTF-8"));
            logData.appendBasicOSCIMessage("rsp2InitDialog", rsp2InitDialog.toString());
            logData.appendBasicOSCIMessage("rsp2FetchDel", rsp2FetchDel.toString());
            logData.appendBasicOSCIMessage("rsp2ExitDialog", rsp2ExitDialog.toString());

            zKSMessageLogger.logEmpfangeneNachricht(contentOfMessage, msgType, logData.getAdresse(), messageId, logData.getOsciMessages());
            //return Content als byte[]
            return contentOfMessage;
        } catch (UnrecoverableKeyException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (OSCIException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (KeyStoreException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (CertificateException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (NoSuchAlgorithmException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (IOException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        }
        return null;

    }

    /**
     * Liefert alle Nachrichten-Header im Postfach des angegebenen Empfängers
     * @param receiverCert
     * @param receiverPassword
     * @return
     */
    public List<MessageHeaderContainer> getMessageList(byte[] receiverCert, String receiverPassword) {
        List<MessageHeaderContainer> containers = new ArrayList<MessageHeaderContainer>();
        try {
            InputStream stream1 = new ByteArrayInputStream(receiverCert);
            InputStream stream2 = new ByteArrayInputStream(receiverCert);
            Originator origi = new Originator(new PKCS12Signer(stream1, receiverPassword), new PKCS12Decrypter(stream2, receiverPassword));
            DialogHandler clientDialog = new DialogHandler(origi, intermed, new HttpTransport());
            InitDialog id = new InitDialog(clientDialog);
            ResponseToInitDialog rsp2InitDialog = null;
            rsp2InitDialog = id.send();
            FetchProcessCard fetchProcCard = new FetchProcessCard(clientDialog);
            fetchProcCard.setRoleForSelection(OSCIMessage.SELECT_ADDRESSEE);
            fetchProcCard.setSelectionMode(OSCIMessage.NO_SELECTION_RULE);
            ResponseToFetchProcessCard rsp2FetchProcCard = fetchProcCard.send();
            ProcessCardBundle[] pBundle = rsp2FetchProcCard.getProcessCardBundles();
            for (ProcessCardBundle processCardBundle : pBundle) {
                MessageHeaderContainer container = new MessageHeaderContainer(processCardBundle.getMessageId(), processCardBundle.getSubject(), OSCIMessage.parseISO8601(processCardBundle.getCreation().getTimeStamp()));
                if (processCardBundle.getReception() != null) {
                    Date reception = OSCIMessage.parseISO8601(processCardBundle.getReception().getTimeStamp());
                    container.setEmpfang(reception);
                }
                containers.add(container);
            }
            ExitDialog exitDialog = new ExitDialog(clientDialog);
            exitDialog.send();
        } catch (ParseException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (OSCIException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (UnrecoverableKeyException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (KeyStoreException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (CertificateException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (NoSuchAlgorithmException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        } catch (IOException ex) {
            Logger.getLogger(OSCIMessageService.class.getName()).log(Level.SEVERE, null, ex);
        }
        return containers;
    }
}
