/**
 * Copyright 2010 FH Trier, Umwelt-Campus Birkenfeld
 * 
 * FH Trier, Umwelt-Campus Birkenfeld  
 * P.O. Box 1380
 * D-55761 Birkenfeld
 * GERMANY
 *
 * http://www.umwelt-campus.de
 *
 * Lizenziert unter der EUPL, Version 1.1 oder - sobald
 * diese von der Europäischen Kommission genehmigt wurden -
 * Folgeversionen der EUPL ("Lizenz");
 * Sie dürfen dieses Werk ausschließlich gemäß dieser Lizenz nutzen.
 * Eine Kopie der Lizenz finden Sie hier:
 *
 * http://ec.europa.eu/idabc/eupl
 *
 * Sofern nicht durch anwendbare Rechtsvorschriften
 * gefordert oder in schriftlicher Form vereinbart, wird
 * die unter der Lizenz verbreitete Software "so wie sie ist",
 * OHNE JEGLICHE GEWÄHRLEISTUNG ODER BEDINGUNGEN -
 * ausdrücklich oder stillschweigend - verbreitet.
 * Die sprachspezifischen Genehmigungen und Beschränkungen
 * unter der Lizenz sind dem Lizenztext zu entnehmen.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package de.umweltcampus.uput.informatik.openeanvadapter.communication.zksservices;

import de.umweltcampus.uput.informatik.openeanvadapter.util.GenericUtils;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyStore;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.crypto.MarshalException;
import javax.xml.crypto.dom.DOMStructure;
import javax.xml.crypto.dsig.CanonicalizationMethod;
import javax.xml.crypto.dsig.DigestMethod;
import javax.xml.crypto.dsig.Reference;
import javax.xml.crypto.dsig.SignatureMethod;
import javax.xml.crypto.dsig.SignedInfo;
import javax.xml.crypto.dsig.Transform;
import javax.xml.crypto.dsig.XMLObject;
import javax.xml.crypto.dsig.XMLSignature;
import javax.xml.crypto.dsig.XMLSignatureException;
import javax.xml.crypto.dsig.XMLSignatureFactory;
import javax.xml.crypto.dsig.dom.DOMSignContext;
import javax.xml.crypto.dsig.keyinfo.KeyInfo;
import javax.xml.crypto.dsig.keyinfo.KeyInfoFactory;
import javax.xml.crypto.dsig.keyinfo.KeyValue;
import javax.xml.crypto.dsig.keyinfo.X509Data;
import javax.xml.crypto.dsig.spec.C14NMethodParameterSpec;
import javax.xml.crypto.dsig.spec.TransformParameterSpec;
import javax.xml.crypto.dsig.spec.XPathFilter2ParameterSpec;
import javax.xml.crypto.dsig.spec.XPathType;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Klasse zum Signieren der Adressnachrichten
 * @author Marc Beck
 */
public class ZKSMessageSigner {

    public static final String xadesNamespace = "http://uri.etsi.org/01903/v1.3.2#";

    /**
     * Signiert eine übergebene Nachricht.
     * @param message die Nachricht selbst (XML-String)
     * @param signatureMetaData Signaturmetadaten
     * @param messageType Nachrichtentyp
     * @return
     */
    public byte[] signMessage(byte[] message, SignatureMetaData signatureMetaData, ZKSMessageType messageType) {
        {
            InputStream is = null;
            try {
                String signedPropertiesID = "SignedProps" + UUID.randomUUID().toString().substring(0, 6);
                //Get DOM SignatureFactory
                XMLSignatureFactory fac = XMLSignatureFactory.getInstance("DOM");

                DigestMethod dm = fac.newDigestMethod(DigestMethod.SHA256, null);
                //Generate the transforms
                List transforms = new ArrayList();
                //SignatureType Transform: Enveloped
                Transform t = fac.newTransform(Transform.ENVELOPED, (TransformParameterSpec) null);
                //Node to be signed Transform Intersect
                XPathType pathTypeIntersect = new XPathType("descendant::*[local-name()='" + messageType.toString() + "']", XPathType.Filter.INTERSECT);
                XPathFilter2ParameterSpec specIntersect = new XPathFilter2ParameterSpec(Collections.singletonList(pathTypeIntersect));
                Transform txpathIntersect = fac.newTransform(Transform.XPATH2, specIntersect);
                //Signature should not be signed: Exclusive
                XPathType pathTypeExclusive = new XPathType("descendant::*[local-name()='" + messageType.toString() + "']/*[local-name()='Signature']", XPathType.Filter.SUBTRACT);
                XPathFilter2ParameterSpec specExclusive = new XPathFilter2ParameterSpec(Collections.singletonList(pathTypeExclusive));
                Transform txpathExclusive = fac.newTransform(Transform.XPATH2, specExclusive);
                transforms.add(t);
                transforms.add(txpathIntersect);
                transforms.add(txpathExclusive);
                //Transform Exclusive
                Transform t1 = fac.newCanonicalizationMethod(CanonicalizationMethod.EXCLUSIVE, ((C14NMethodParameterSpec) null));
                transforms.add(t1);
                List<Reference> refList = new ArrayList<Reference>();
                refList.add(fac.newReference("", dm, transforms, null, null));

                refList.add(fac.newReference("#" + signedPropertiesID, dm, Collections.singletonList(t1), xadesNamespace + "SignedProperties", null));

                CanonicalizationMethod cn = fac.newCanonicalizationMethod(CanonicalizationMethod.EXCLUSIVE, (C14NMethodParameterSpec) null);

                SignatureMethod sm = fac.newSignatureMethod(SignatureMethod.RSA_SHA1, null);//XXX In Zukünftigen Versionen auf SHA256 wechseln
                SignedInfo si = fac.newSignedInfo(cn, sm, refList);
                //Neuer Keystore im PKCS12-Format
                KeyStore myKeystore = KeyStore.getInstance("PKCS12");
                is = new ByteArrayInputStream(signatureMetaData.getPrivateCert());
                //KeyStore laden
                myKeystore.load(is, signatureMetaData.getPrivateCertPassword().toCharArray());
                //PrivateKey
                KeyStore.PasswordProtection pwd = new KeyStore.PasswordProtection(signatureMetaData.getPrivateCertPassword().toCharArray());
                Enumeration<String> e = myKeystore.aliases();
                //Alias
                String alias = "";
                while (e.hasMoreElements()) {
                    alias = e.nextElement();
                }
                KeyStore.PrivateKeyEntry thePrivKeyEntry = (KeyStore.PrivateKeyEntry) myKeystore.getEntry(alias, pwd);
                X509Certificate cert = (X509Certificate) thePrivKeyEntry.getCertificate();
                // Create the X509Data.
                KeyInfoFactory kif = fac.getKeyInfoFactory();
                List x509Content = new ArrayList();
                x509Content.add(cert.getSubjectX500Principal().getName());
                x509Content.add(cert);
                X509Data xd = kif.newX509Data(x509Content);
                //Create the KeyInfo with KeyValue and X509Data
                KeyValue keyValue = kif.newKeyValue(cert.getPublicKey());
                List keyInfoList = new ArrayList();
                keyInfoList.add(xd);
                keyInfoList.add(keyValue);
                KeyInfo ki = kif.newKeyInfo(keyInfoList);

                // Instantiate the document to be signed
                DocumentBuilderFactory docBuildFactory = DocumentBuilderFactory.newInstance();
                docBuildFactory.setNamespaceAware(true);
                Document doc = docBuildFactory.newDocumentBuilder().parse(new InputSource(new ByteArrayInputStream(message)));

                // Create a DOMSignContext and specify the DSA PrivateKey and
                // location of the resulting XMLSignature's parent element
                XPath xPath = XPathFactory.newInstance().newXPath();
                String xPathString = "/descendant::*[local-name()='" + messageType.toString() + "']";
                Node node = (Node) xPath.evaluate(xPathString, doc, XPathConstants.NODE);
                DOMSignContext dsc = new DOMSignContext((Key) thePrivKeyEntry.getPrivateKey(), node);
                // Namespace-Prefix for the Signature "ds"
                dsc.putNamespacePrefix(XMLSignature.XMLNS, "ds");
                //SignaturId: Setzt sich lt. Schema aus Rolle und UUID zusammen
                String signatureId = signatureMetaData.getXadesPrefix() + "-" + UUID.randomUUID().toString();
                XMLObject object = null;
                //XADES Redefine (siehe Spezifikation 1.4 S.19-20)
                Element qualifyingProperties = buildXAdESStructure(doc, signedPropertiesID, signatureId);
                //Object tag
                object = fac.newXMLObject(Collections.singletonList(new DOMStructure(qualifyingProperties)), null, null, null);
                // Create the XMLSignature (but don't sign it yet)
                XMLSignature signature = fac.newXMLSignature(si, ki, Collections.singletonList(object), signatureId, null);

                // Marshal, generate (and sign) the enveloped signature
                signature.sign(dsc);

                fac = null;
                TransformerFactory tf = TransformerFactory.newInstance();
                Transformer trans = tf.newTransformer();
                trans.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                trans.transform(new DOMSource(doc), new StreamResult(baos));
                return (baos.toByteArray());


            } catch (GeneralSecurityException ex) {
                Logger.getLogger(ZKSMessageSigner.class.getName()).log(Level.SEVERE, null, ex);
            } catch (XPathExpressionException ex) {
                Logger.getLogger(ZKSMessageSigner.class.getName()).log(Level.SEVERE, null, ex);
            } catch (ParserConfigurationException ex) {
                Logger.getLogger(ZKSMessageSigner.class.getName()).log(Level.SEVERE, null, ex);
            } catch (SAXException ex) {
                Logger.getLogger(ZKSMessageSigner.class.getName()).log(Level.SEVERE, null, ex);
            } catch (TransformerException ex) {
                Logger.getLogger(ZKSMessageSigner.class.getName()).log(Level.SEVERE, null, ex);
            } catch (MarshalException ex) {
                Logger.getLogger(ZKSMessageSigner.class.getName()).log(Level.SEVERE, null, ex);
            } catch (XMLSignatureException ex) {
                Logger.getLogger(ZKSMessageSigner.class.getName()).log(Level.SEVERE, null, ex);
            } catch (IOException ex) {
                Logger.getLogger(ZKSMessageSigner.class.getName()).log(Level.SEVERE, null, ex);
            } finally {
                try {
                    is.close();
                } catch (IOException ex) {
                    Logger.getLogger(ZKSMessageSigner.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
        }
        return null;
    }

    /**
     * Liefert die XAdES-T Struktur.<br>
     * <xad:QualifyingProperties Target="value" xmlns:xad="http://uri.etsi.org/01903/v1.3.2#">
     *       <xad:SignedProperties Id="SignedPropertiesId">
     *         <xad:SignedSignatureProperties>
     *           <xad:SigningTime>value</xad:SigningTime>
     *         </xad:SignedSignatureProperties>
     *       </xad:SignedProperties>
     *     </xad:QualifyingProperties>
     * @param doc
     * @param certificate
     * @param signedPropertiesID
     * @param signatureId
     * @return
     * @throws GeneralSecurityException
     */
    private Element buildXAdESStructure(Document doc, String signedPropertiesID, String signatureId) throws GeneralSecurityException {
        //XADES Redefine (siehe Spezifikation 1.4 S.19-20)
        Element qualifyingProperties = doc.createElementNS(xadesNamespace, "xades:QualifyingProperties");
        qualifyingProperties.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns:" + "xades", xadesNamespace);
        Element signedProperties = doc.createElementNS(xadesNamespace, "xades:SignedProperties");
        signedProperties.setAttribute("Id", signedPropertiesID);
        //signedProperties.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns", xadesNamespace);
        Element signedSignatureProperties = doc.createElementNS(xadesNamespace, "xades:SignedSignatureProperties");
        //ISO 8601 (Atom) Timestamp
        Element signingTime = doc.createElementNS(xadesNamespace, "xades:SigningTime");
        String time = GenericUtils.getISO8601Timestamp(new Date());
        signingTime.appendChild(doc.createTextNode(time));
        signedSignatureProperties.appendChild(signingTime);
        signedProperties.appendChild(signedSignatureProperties);
        qualifyingProperties.appendChild(signedProperties);
        qualifyingProperties.setAttribute("Target", signatureId);
        return qualifyingProperties;
    }

    public SignatureMetaData buildMetaData(byte[] privateCert, String privateCertPassword, String xadesPrefix) {
        return new SignatureMetaData(privateCert, privateCertPassword, xadesPrefix);
    }

    public class SignatureMetaData {

        private final byte[] privateCert;
        private final String privateCertPassword;
        private final String xadesPrefix;

        public SignatureMetaData(byte[] privateCert, String privateCertPassword, String xadesPrefix) {
            this.privateCert = privateCert;
            this.privateCertPassword = privateCertPassword;
            this.xadesPrefix = xadesPrefix;
        }

        public byte[] getPrivateCert() {
            return privateCert;
        }

        public String getPrivateCertPassword() {
            return privateCertPassword;
        }

        public String getXadesPrefix() {
            return xadesPrefix;
        }
    }
}
